﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common.Extensions;
using VA.PPMS.IWS.MappingService.Helpers;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapBoardCertificationToCrm : MapperRelatedBase
    {
        public MapBoardCertificationToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper) 
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public async Task<ppms_boardcertification> MapInsert(BoardCertification certification, Account provider)
        {
            // map entity
            var entity = new ppms_boardcertification
            {
                Id = Guid.NewGuid(),
                ppms_name = certification.Name,
                ppms_boardid = certification.BoardId
            };

            if (certification.IsBoardCertifiedSpecified)
                entity.ppms_isboardcertified = certification.IsBoardCertified;

            if (certification.CertificationDateSpecified && certification.CertificationDate.Year > EnumHelper.MinYear)
                entity.ppms_certificationdate = certification.CertificationDate.ToCrmDate();

            if (certification.RecertificationDateSpecified && certification.RecertificationDate.Year > EnumHelper.MinYear)
                entity.ppms_recertificationdate = certification.RecertificationDate.ToCrmDate();

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) entity.OwnerId = Owner;

            await Task.Run(() => { });

            return entity;
        }

        public async Task<ppms_boardcertification> MapUpdate(BoardCertification entity, Account provider)
        {
            // get entity
            var crmEntity = GetCrmEntity(provider, entity.BoardId);

            // matching account not found
            if (crmEntity == null)
            {
                return await MapInsert(entity, provider);
            }

            // map fields
            var newEntity = new ppms_boardcertification()
            {
                Id = crmEntity.Id,
                ppms_name = crmEntity.ppms_name,
                ppms_isboardcertified = crmEntity.ppms_isboardcertified,
                ppms_certificationdate = crmEntity.ppms_certificationdate,
                ppms_recertificationdate = crmEntity.ppms_recertificationdate
            };

            if (IsChanged(entity.Name, crmEntity.ppms_name))
                newEntity.ppms_name = entity.Name;

            if (entity.IsBoardCertifiedSpecified)
                newEntity.ppms_isboardcertified = entity.IsBoardCertified;

            if (entity.CertificationDateSpecified && entity.CertificationDate.Year > EnumHelper.MinYear)
                newEntity.ppms_certificationdate = entity.CertificationDate.ToCrmDate();

            if (entity.RecertificationDateSpecified && entity.RecertificationDate.Year > EnumHelper.MinYear)
                newEntity.ppms_recertificationdate = entity.RecertificationDate.ToCrmDate();

            // Set owner to CCN
            if (!ForVaNetwork && Owner != null) newEntity.OwnerId = Owner;

            return newEntity;
        }

        private static ppms_boardcertification GetCrmEntity(Account provider, string id)
        {
            var list = provider.ppms_account_ppms_boardcertification;
            if (list == null) return null;

            var ppmsBoardcertifications = list as ppms_boardcertification[] ?? list.ToArray();

            return ppmsBoardcertifications.Any() ? ppmsBoardcertifications.FirstOrDefault(x => x.ppms_boardid == id) : null;
        }

        private static BoardCertification ConvertEntity<T>(T entity)
        {
            return (BoardCertification)Convert.ChangeType(entity, typeof(BoardCertification));
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (IsWithinContext) AssociateRelatedEntities(parent, entities, "ppms_account_ppms_boardcertification");
            else
            {
                if (entities != null && entities.Count > 0)
                {
                    var account = (Account)parent;
                    if (account != null)
                        account.ppms_account_ppms_boardcertification = ConvertEntityList<ppms_boardcertification>(entities);
                }
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<ppms_boardcertification>();

            // Map schema entities for delete
            foreach (var item in entities)
            {
                var entity = ConvertEntity(item);
                var matches = provider.ppms_account_ppms_boardcertification.Where(p => p.ppms_name == entity.Name);
                list.AddRange(matches);
            }

            return EntityDelete((IEnumerable<ppms_boardcertification>)list);
        }
    }
}